package com.burhanstore.neonbubblequest.act

import android.app.Activity
import android.content.Intent
import android.os.Bundle
import android.util.Log
import android.view.View
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.burhanstore.neonbubblequest.AppConst
import com.burhanstore.neonbubblequest.MainActivity
import com.burhanstore.neonbubblequest.Utils.AppString
import com.burhanstore.neonbubblequest.Utils.anim.AnimationHelper
import com.burhanstore.neonbubblequest.Utils.appuser.UserManager
import com.burhanstore.neonbubblequest.Utils.network.AppNetworkAdapter
import com.burhanstore.neonbubblequest.Utils.response.SignupResponse
import com.burhanstore.neonbubblequest.databinding.ActivityLoginBinding
import com.google.android.gms.auth.api.signin.GoogleSignIn
import com.google.android.gms.auth.api.signin.GoogleSignInClient
import com.google.android.gms.auth.api.signin.GoogleSignInOptions
import com.google.android.gms.common.api.ApiException
import com.google.gson.Gson
import com.google.gson.JsonObject
import com.google.gson.JsonParser
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

class LoginActivity : AppCompatActivity() {


    private lateinit var binding: ActivityLoginBinding
    private lateinit var user_m: UserManager
    private lateinit var activity: Activity
    private lateinit var googleSignInClient: GoogleSignInClient
    private val prefs by lazy { activity.getSharedPreferences("app_prefs", MODE_PRIVATE) }
    var f_token: String? = null

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = ActivityLoginBinding.inflate(layoutInflater)
        setContentView(binding.root)
        activity = this

        user_m = UserManager.getInstance(activity)
        AppString.getFcmToken { token ->
            if (token.isNotEmpty()) {
                println("FCM Token: $token")
                f_token = token
            }
        }

        binding.cbPrivacy.isChecked = prefs.getBoolean("privacy_accepted", false)
        binding.cbPrivacy.setOnClickListener {
            val newAccepted = binding.cbPrivacy.isChecked
            prefs.edit().putBoolean("privacy_accepted", newAccepted).apply()
        }

        int_firebase()

        binding.conGoogle.setOnClickListener {
            if (!isPrivacyAccepted()) {
                Toast.makeText(
                    activity,
                    "You must accept the Privacy Policy to sign up",
                    Toast.LENGTH_SHORT
                ).show()
                return@setOnClickListener
            }
            googleSignIn()
        }

        binding.guestLogin.setOnClickListener {
            regester_user(
                "Guest",
                "guest@gmail.com",
                ""
            )
        }


    }

    // Helper to check latest privacy acceptance
    private fun isPrivacyAccepted(): Boolean {
        return prefs.getBoolean("privacy_accepted", false)
    }

    // Encapsulated Google Sign-In logic
    private fun googleSignIn() {
        val signInIntent = googleSignInClient.signInIntent
        signInLauncher.launch(signInIntent)
    }

    private fun int_firebase() {
        val gso =
            GoogleSignInOptions.Builder(GoogleSignInOptions.DEFAULT_SIGN_IN).requestEmail().build()
        googleSignInClient = GoogleSignIn.getClient(this, gso)

    }

    private val signInLauncher =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) { result ->
            val task = GoogleSignIn.getSignedInAccountFromIntent(result.data)
            try {
                val account = task.getResult(ApiException::class.java)
                binding.Loanimation.visibility = View.VISIBLE
                binding.loginLy.visibility = View.GONE

                regester_user(
                    account?.displayName ?: "",
                    account?.email ?: "",
                    account?.photoUrl.toString()
                )


            } catch (e: ApiException) {
                e.printStackTrace()
                Toast.makeText(activity, "Failed to get email", Toast.LENGTH_SHORT).show()
            }

        }


    private fun regester_user(name: String, email: String, profile: String) {

        val json = JsonObject().apply {
            addProperty("action", "register")
            addProperty("f_token", f_token)
            addProperty("name", name)
            addProperty("image", profile)
            addProperty("email", email)
            addProperty("android_id", AppString.getAndroidId(activity))
            addProperty("u_id", AppString.generateUid())
            addProperty("refer", AppString.generateReferId())
            addProperty("signup_type", "Google")
            addProperty("referred_by", binding.referredBy.getText().toString())
            addProperty("country_name", AppString.getCountry(this@LoginActivity))
        }

        AppNetworkAdapter.sendRequest(
            lifecycleScope,
            json,
            object : AppNetworkAdapter.ResponseHandler {
                override fun onSuccess(response: JsonObject?) {
                    Log.d("LoginActivity", "Success: $response")

                    lifecycleScope.launch {
                        val signupResponse = Gson().fromJson(response, SignupResponse::class.java)
                        val firstUser = signupResponse.user?.firstOrNull()
                        val setting = signupResponse.setting

                        if (firstUser != null) {
                            user_m.saveUser(firstUser)
                            user_m.saveSetting(setting)

                            withContext(Dispatchers.Main) {


                                AppConst.setString(
                                    activity, AppConst.USER_COIN,
                                    user_m.getUser()?.point ?: "0"
                                )

                                val intent = Intent(activity, MainActivity::class.java)
                                startActivity(intent)
                                AnimationHelper.applySlideZoomTransition(activity)
                                finish()
                                Toast.makeText(activity, signupResponse.message, Toast.LENGTH_SHORT)
                                    .show()

                            }


                        }
                    }


                }

                override fun onError(code: Int, message: String?) {
                    Log.e("LoginActivity", "Error $code: $message")
                    binding.Loanimation.visibility = View.GONE
                    binding.loginLy.visibility = View.VISIBLE

                    try {
                        message?.let {
                            val json = JsonParser.parseString(it).asJsonObject
                            val errorMsg = json.get("message")?.asString ?: "Unknown error"
                            Toast.makeText(activity, errorMsg, Toast.LENGTH_SHORT).show()
                        }
                    } catch (e: Exception) {
                        e.printStackTrace()
                        Toast.makeText(activity, "Unknown error occurred", Toast.LENGTH_SHORT)
                            .show()
                    }

                }

                override fun onException(exception: Exception) {
                    Log.e("LoginActivity", "Exception: ", exception)
                    binding.Loanimation.visibility = View.GONE
                    binding.loginLy.visibility = View.VISIBLE
                }
            })

    }

}